
/*
  CLASSiC DAC, Copyright 2021 SILICON CHIP Publications
  main.c: Main loop and control routines
  Written by Nicholas Vinen, 2012-2021

  v1.01, August 2021, changes:
  * Rewrite pushbutton & AC input sensing to provide much better button debouncing and avoid switching off unless button is held in for a significant time.
  * Changes to IR reception code to reject noise and prevent accidental power on/off. Aborts IR processing if a corrupted signal is detected and two "standby" IR messages need to be received in succession before it will power on or off.
  * Change to audio buffering - don't advance buffer pointed during underruns to prevent noise and glitches when changing tracks/directories/etc.
*/

#include "p33Fxxxx.h"
#include "Timer.h"
#include "Control_SPI.h"
#include "PLL1708.h"
#include "CS8416.h"
#include "CS4398.h"
#include "DCI.h"
#include "Clock.h"
#include "Infrared.h"
#include "Biphase.h"
#include "Playback.h"
#include "Config.h"
#include "DEE Emulation 16-bit.h"
#include "sdcard/ff.h"
#include "sdcard/spi.h"
#include <string.h>

// Configuration registers
_FGS(GWRP_OFF & GCP_OFF);                        // We don't want to write protect the flash as we use it to store state.
_FOSCSEL(FNOSC_FRC);                             // Start up initially without the PLL but instead using the internal fast RC oscillator.
_FOSC(FCKSM_CSECMD & OSCIOFNC_ON & POSCMD_NONE); // Clock switching enabled, no clock output, no crystal driver.
_FWDT(FWDTEN_OFF);                               // Not using the watchdog timer.
#ifdef __DEBUG
_FICD(ICS_PGD3&JTAGEN_OFF);                      // Debug on ICSP port.
#endif

/*
  chip: dsPIC33FJ128GP306
  clock: 7.37MHz [FRC]/39.6MHz [FRCPLL]
  pins:

    RB0  (16): LED1 1.2k cathode
    RB1  (15): LED1 680R cathode
    RB2  (14): LED2 1.2k cathode
    RB3  (13): LED2 680R cathode
    RB4  (12): LED3 1.2k cathode
    RB5  (11): LED3 680R cathode
    RB6  (17): LED4 820R cathode
    RB7  (18): LED4 470R cathode
    RB8  (21): LED5 1.2k cathode
    RB9  (22): LED5 680R cathode
    RB10 (23): LED6 1.2k cathode
    RB11 (24): LED6 680R cathode
    RB12 (27): LED7 1.2k cathode
    RB13 (28): LED7 680R cathode
    RB14 (29): LED8 2.2k cathode
    RB15 (30): LED8 1.2k cathode

    RC1  ( 2): power control (active high) (controls +15V, -15V & TOSLINK supplies)
    RC2  ( 3): CS8416 CS
    RC12 (39): Sample rate LEDs anode #1 (??)
  * RC13 (47): CS8416 GPO2 [CN1]
    RC14 (48): CS4398 RST
    RC15 (40): Sample rate LEDs anode #2 (??)

    RD0  (46): PLL1708 CS
    RD1  (49): CS4398 CS
    RD2  (50): CS8416 RST
    RD3  (51): Headphone socket switch (closed = low)
  * RD4  (52): CS4398 LMUTEC
    RD5  (53): CS8416 GPO1 [CN14]
    RD6  (54): PCM2902 SSPND
    RD7  (55): Power switch/AC sense
    RD8  (42): Sample rate LEDs cathode 4.7k# (??)
    RD9  (43): Sample rate LEDs anode #3 (??)
    RD10 (44): Sample rate LEDs anode #4 (??)
    RD11 (45): Sample rate LEDs cathode 2.2k# (??)

    RF0  (58): DIP switch #1
    RF1  (59): DIP switch #2
    RF2  (34): SD card DATA OUT
    RF3  (33): SD card DATA IN
    RF4  (31): SD card detect (present = low) [CN17]
    RF5  (32): SD card CS
    RF6  (35): SD card CLK

    RG0  (61): Power LED (active low)
    RG1  (60): DIP switch #3
    RG2  (37): SD card +3.3V power enable (active low)
    RG3  (36): SD card write protect (protected = low)
    RG6  ( 4): Control bus serial clock [SCK2]
    RG7  ( 5): Control bus data input [SDI2]
    RG8  ( 6): Control bus serial data  [SDO2]
    RG9  ( 8): Infrared data input [CN11]
    RG12 (63): N/C
    RG13 (64): CS8416 RXP7 (Input #7) [CSDO]
    RG14 (62): PLL1708 SCKO2 / 2 [CSCK]
    RG15 ( 1): N/C

  * = not used

  Sampling rate LED states: (* = bright, o = dim, x = alternating, ! = flashing)
                  44.1   48   96  192
  Unlocked
  44.1k no audio     o
  44.1k    audio     *
  44.1k    de-emph   x
  48.0k no audio          o
  48.0k    audio          *
  48.0k    de-emph        x
  88.2k no audio          o    o
  88.2k    audio          *    *
  96.0k no audio               o
  96.0k    audio               *
 176.4k no audio               o    o
 176.4k    audio               *    *
 192.0k no audio                    o
 192.0k    audio                    *
  other no audio     o    o
  other    audio     *    *
  other    de-emph   x    x
   DTS (muted)       !
   AC-3 (muted)           !
  Other (muted)                !     
  Error (muted)                     !

  Remote control functions:

    <normal mode>
    Power: standby toggle
    Mute: mute toggle
    1-8: select input
    0: (bass/treble) toggle fixed scale
    9: N/A
    AV: N/A
    -/--: N/A
    CHUP, CHDN: (normal) pan right/left (bass/treble) adjust bass crossover frequency (crossfeed) change attenuation 
    VOLUP, VOLDN: (normal, crossfeed) change volume (bass/treble) adjust treble crossover frequency
    Left/right: (normal) previous/next track (bass/treble) adjust treble boost/cut (crossfeed) adjust crossfeed crossover frequency
    Up/down: (normal) previous/next folder (bass/treble) adjust bass boost/cut (crossfeed) adjust crossfeed delay
    OK: (normal) cycle through playback orders (bass/treble) reset controls (crossfeed) reset controls
    Teletext: go to tone control mode
    Pagehold: go to crossfeed control mode
    TV/Video: go to normal mode
    OSD: (bass/treble) turn on auto-loudness (crossfeed) toggle crossfeed
    Record: (normal) pan center (bass/treble) toggle tone control enable flag
    Pause: toggle pause
    Exit: N/A
    rw/ff: skip back/forward 10 seconds
    play: play
    stop: stop(/reset?)
*/

//////////// Functions and variables for controlling power to SD card and determining SD card interface state ////////////

unsigned char memoryCardSystemUp; // flag indicates whether SD card has been detected & configured
static void SetSDCardPower(bool bPowered) {
  DeinitSPI();
  LATGbits.LATG2 = !bPowered;
  TRISGbits.TRISG2 = 0;
}
static void InitSDCardDetect() {
  TRISFbits.TRISF4 = 1;
  CNPU2bits.CN17PUE = 1;
}

//////////// Functions and variables for dealing with power button and AC power monitoring ////////////

volatile unsigned char poweron, power_button_pressed = 0, power_button_press_cycles = 0, g_ignore_power_button = 0;
unsigned char disable_power_loss_quick_sense;
void SaveSettingsToFlash();

// function which is called if a loss of AC power is detected, to save settings & shut down cleanly
static void PowerLost(unsigned char reason) {
//#ifndef __DEBUG
  // AC power lost - turn off LEDs to save power so caps don't discharge so fast
  // also drop CPU rate
  TRISB = 0xFFFF;
  LATDbits.LATD10 = 1;
  LATDbits.LATD9  = 1;
  LATCbits.LATC15 = 1;
  LATCbits.LATC12 = 1;
  TRISGbits.TRISG0 = 1;
//  Clock_SetSpeed(false);
  // now save settings
  SaveSettingsToFlash();
  poweron = 0;
//#endif
}

// Timer 5 expiry indicates AC crossing from plugpack not detected in the expected timeframe.
// This could indicate either loss of AC power or a press of the power button.
void __attribute__((__interrupt__,no_auto_psv)) _T5Interrupt( void ) {
  if( g_ignore_power_button )
    --g_ignore_power_button;
  if( power_button_pressed ) {
    ++power_button_press_cycles;
  } else {
    power_button_press_cycles = 0;
  }

  if( !power_button_pressed ) {
    if( !PORTDbits.RD7 ) {
      CNPU2bits.CN16PUE = 1;
      Nop();
      Nop();
      if( !PORTDbits.RD7 ) {
        power_button_pressed = 1;
        TMR5 = 0;
      } else {
        CNPU2bits.CN16PUE = 0;
        PowerLost(1);
      }
    } else {
      PowerLost(2);
    }
  }

  IFS1bits.T5IF = 0;
}

static void ACSense_Init() {
  CNEN2bits.CN16IE = 1;
  power_button_pressed = 0;
  power_button_press_cycles = 0;
  disable_power_loss_quick_sense = 0;

  TMR5 = 0;
  PR5 = 60000;//55000;
  IFS1bits.T5IF = 0;
  IEC1bits.T5IE = 1;
  T5CONbits.TCKPS = 1;
  T5CONbits.TON = 1;
}

static void ACSense_Deinit() {
  asm("#disi 16");
  CNEN2bits.CN16IE = 0;
  T5CONbits.TON = 0;
  IEC1bits.T5IE = 0;
  T5CONbits.TON = 0;
  asm("#disi 0");
}

//////////// Functions to handle clock generator and sampling rate calculations ////////////

static bool Test_PLL1708() {
  unsigned int i;
  unsigned long freq, measured, expected;

  Timer_Delay_ms(30);
  for( i = 0; i < PLL1708_NUM_VALID_FREQUENCIES; ++i ) {
    freq = PLL1708_valid_frequencies[i];
    PLL1708_Configure(freq);
    Timer_Delay_ms(20);
    measured = DCI_Measure_Clock_Frequency(20);
    expected = freq * 128UL;
    if( measured < (expected - expected / 16) || measured > (expected + expected / 16) )
      return false;
  }
  PLL1708_Configure(0);
  return true;
}

#define NUM_POSSIBLE_ESTIMATED_RATES 15
static const unsigned long g_PossibleEstimatedRates[NUM_POSSIBLE_ESTIMATED_RATES] = { 8000UL, 11025UL, 12000UL, 16000UL, 22050UL, 24000UL, 32000UL, 44100UL, 48000UL, 64000UL, 88200UL, 96000UL, 128000UL, 176400UL, 192000UL };
static unsigned long EstimatePlaybackSampleRate(unsigned long* RawEstimate) {
  unsigned char ratio = CS8416_Read_Register(CS8416_REG_OMCK_RMCK_RATIO);
  unsigned long estimate = (PLL1708_Get_Current_Rate() << 6) / ratio;
  unsigned long estimate_low = estimate * 15 / 16, estimate_high = estimate * 17 / 16;
  unsigned char i;

  if( RawEstimate )
    *RawEstimate = estimate;

  for( i = 0; i < NUM_POSSIBLE_ESTIMATED_RATES; ++i ) {
    if( g_PossibleEstimatedRates[i] < estimate_high && g_PossibleEstimatedRates[i] > estimate_low )
      return g_PossibleEstimatedRates[i];
  }

  return 0;
}


//////////// Functions and variable for dealing with device configuration and resume from standby/power off ////////////
//////////// Also included are functions to set DAC & digital audio receiver state to suit current configuration ////////////

// Configuration & state variables
DACConfig g_Config;
playback_state_t resume_state;
char resume_path[_MAX_LFN+20];
unsigned long resume_pos;
unsigned char g_CurrentVolumeAdjust, muted, g_CurrentRMCKF;

static void CS4398_Set_Channel_Config() {
  CS4398_Write_Register(CS4398_REG_MIX, (g_Config.DownmixToMono ? CS4398_MIX_MONO : g_Config.StereoSwap ? CS4398_MIX_SWAP_LR : CS4398_MIX_NORMAL)|(g_Config.InvertPolarity ? CS4398_MIX_INVERT_POL : 0));
}

static void CS4398_Set_Filter_Config() {
  CS4398_Write_Register(CS4398_REG_RAMPFILTER, CS4398_RAMP_SOFTRAMP|CS4398_RAMP_UP_ERROR|CS4398_RAMP_DN_ERROR|(g_Config.Filter_Slowrolloff ? CS4398_RAMP_SLOW_FILT : 0));
}

static void CS8416_Update_Deemph_Config() {
  CS8416_Write_Register(CS8416_REG_CONTROL2, (CS8416_GPO_TX<<CS8416_CONTROL2_GPO0SEL_SHIFT) | ((g_Config.NoDeEmphasis ? CS8416_CONTROL2_EMPH_OFF : CS8416_CONTROL2_EMPH_AUTO)<<CS8416_CONTROL2_EMPH_CNTL_SHIFT));
}

static void CS8416_Update_RMCKF_and_SWCLK() {
  CS8416_Write_Register(CS8416_REG_CONTROL1, (g_CurrentRMCKF ? CS8416_CONTROL1_RMCKF : 0)|(g_Config.NoFreeRunningPLL ? CS8416_CONTROL1_SWCLK : 0));
}

static void UpdateMuteStatus() {
  unsigned char rcvr_err = CS8416_Read_Register(CS8416_REG_RCVR_ERROR) != 0;
  unsigned char non_pcm = !(CS8416_Read_Register(CS8416_REG_AUDIO_FORMAT_DETECT) & CS8416_AUDIO_FORMAT_PCM);
  CS4398_Write_Register(CS4398_REG_MUTEC, (CS4398_MUTEC_MUTEP_AUTO<<CS4398_MUTEC_MUTEP_SHIFT)|(rcvr_err || non_pcm || muted ? (CS4398_MUTEC_MUTE_A|CS4398_MUTEC_MUTE_B) : 0)|CS4398_MUTEC_MUTECAB);
}

void SaveSettingsToFlash() {
  unsigned char idx = 0;

  resume_state = memoryCardSystemUp ? Playback_Get_Status() : reset;
  DataEEWrite(g_Config.Input, ++idx);
  DataEEWrite(g_Config.Volume, ++idx);
  DataEEWrite(g_Config.Balance, ++idx);
  DataEEWrite(g_Config.Tone_Enabled, ++idx);
  DataEEWrite(g_Config.Tone_AutoLoudness, ++idx);
  DataEEWrite(g_Config.Tone_BassBoostCut, ++idx);
  DataEEWrite(g_Config.Tone_TrebleBoostCut, ++idx);
  DataEEWrite(g_Config.Tone_BassCrossoverFreq, ++idx);
  DataEEWrite(g_Config.Tone_TrebleCrossoverFreq, ++idx);
  DataEEWrite(g_Config.Crossfeed_Enabled, ++idx);
  DataEEWrite(g_Config.Crossfeed_LPFFreq, ++idx);
  DataEEWrite(g_Config.Crossfeed_Delay, ++idx);
  DataEEWrite(g_Config.Crossfeed_Atten, ++idx);
  DataEEWrite(resume_state, ++idx);
  if( resume_state == playing || resume_state == paused || resume_state == stopped ) {
    unsigned int i, len;
    if( Playback_Get_Current_Position(&resume_pos) != ok )
      resume_pos = 0;
    if( Playback_Get_Current_File(resume_path, sizeof(resume_path)) != ok )
      strcpy(resume_path, "0:\\");
    DataEEWrite(resume_pos, ++idx);
    DataEEWrite(resume_pos>>16, ++idx);
    len = strlen(resume_path);
    DataEEWrite(len, ++idx);
    len = (len+1)>>1;
    for( i = 0; i < len; ++i ) {
      DataEEWrite(((unsigned short*)resume_path)[i], ++idx);
    }
  }
}

unsigned int DataEEReadDef(unsigned int def, unsigned int addr) {
  unsigned int ret = DataEERead(addr);
  if( ret == 0xFFFF )
    ret = def;
  return ret;
}

static unsigned char UpdateBassTreble() {
  unsigned char db_atten;
  if( g_Config.Tone_AutoLoudness ) {
    g_Config.Tone_BassCrossoverFreq = 500;
    g_Config.Tone_TrebleCrossoverFreq = 5000;
    g_Config.Tone_BassBoostCut = g_Config.Volume / 4;
    g_Config.Tone_TrebleBoostCut = g_Config.Volume / 8;
    if( g_Config.Tone_BassBoostCut > 12 )
      g_Config.Tone_BassBoostCut = 12;
    if( g_Config.Tone_TrebleBoostCut > 8 )
      g_Config.Tone_TrebleBoostCut = 8;
  }
  db_atten = DCI_Set_Filtering(g_Config.Tone_Enabled, g_Config.Tone_BassCrossoverFreq, g_Config.Tone_TrebleCrossoverFreq, ((signed short)g_Config.Tone_BassBoostCut) * 2000, ((signed short)g_Config.Tone_TrebleBoostCut) * 2000, g_Config.Tone_EqualVolume || g_Config.Tone_AutoLoudness);
  if( g_Config.Tone_Enabled && g_Config.Tone_AutoLoudness )
    db_atten -= 18;
  if( db_atten != g_CurrentVolumeAdjust ) {
    g_CurrentVolumeAdjust = db_atten;
    CS4398_Set_Volume_and_Balance(g_Config.Volume + g_CurrentVolumeAdjust, g_Config.Balance);
    return 1;
  }
  return 0;
}

static void UpdateCrossfeed() {
  Playback_Set_Crossfeed(g_Config.Crossfeed_Enabled && (g_Config.Crossfeed_IgnoreHPSocket || !g_Config.Crossfeed_Disabled), g_Config.Crossfeed_LPFFreq, g_Config.Crossfeed_Delay, g_Config.Crossfeed_Atten);
}

void LoadSettingsFromFlash() {
  unsigned char idx = 0;
  g_Config.Input = DataEEReadDef(g_Config.Input, ++idx);
  g_Config.Volume = DataEEReadDef(g_Config.Volume, ++idx);
  g_Config.Balance = DataEEReadDef(g_Config.Balance, ++idx);
  g_Config.Tone_Enabled = DataEEReadDef(g_Config.Tone_Enabled, ++idx);
  g_Config.Tone_AutoLoudness = DataEEReadDef(g_Config.Tone_AutoLoudness, ++idx);
  g_Config.Tone_BassBoostCut = (char)DataEEReadDef(g_Config.Tone_BassBoostCut, ++idx);
  g_Config.Tone_TrebleBoostCut = (char)DataEEReadDef(g_Config.Tone_TrebleBoostCut, ++idx);
  g_Config.Tone_BassCrossoverFreq = (char)DataEEReadDef(g_Config.Tone_BassCrossoverFreq, ++idx);
  g_Config.Tone_TrebleCrossoverFreq = (char)DataEEReadDef(g_Config.Tone_TrebleCrossoverFreq, ++idx);
  g_Config.Crossfeed_Enabled = DataEEReadDef(g_Config.Crossfeed_Enabled, ++idx);
  g_Config.Crossfeed_LPFFreq = DataEEReadDef(g_Config.Crossfeed_LPFFreq, ++idx);
  g_Config.Crossfeed_Delay = DataEEReadDef(g_Config.Crossfeed_Delay, ++idx);
  g_Config.Crossfeed_Atten = DataEEReadDef(g_Config.Crossfeed_Atten, ++idx);
  resume_state = DataEEReadDef(resume_state, ++idx);
  if( resume_state == playing || resume_state == paused || resume_state == stopped ) {
    unsigned int i, len;
    resume_pos = DataEEReadDef(0xFFFF, ++idx);
    resume_pos += (((unsigned long)DataEEReadDef(0xFFFF, ++idx))<<16);
    if( resume_pos == 0xFFFFFFFFUL )
      resume_pos = 0;
    len = DataEEReadDef(0, ++idx);
    if( len && len < sizeof(resume_path) ) {
      for( i = 0; i < len; i += 2 )
        ((unsigned short*)resume_path)[i>>1] = DataEEReadDef(0, ++idx);
      resume_path[len] = '\0';
    }
  }

  UpdateBassTreble();
  UpdateCrossfeed();
}

static unsigned char lfn_to_sfn(char* dest, char* src, unsigned int max_dest_len) {
  char* ptr;
  DIR dir;
  FILINFO file;
  unsigned int len;
  char temp;

  if( max_dest_len < 3 )
    return 0;
  memcpy(dest, src, 3);
  dest += 3;
  max_dest_len -= 3;
  ptr = src + 3;
  while( *ptr != '\0' ) {
    char* start = ptr;
    while( *ptr != '\0' && *ptr != '\\' && *ptr != '/' )
      ++ptr;
    if( start - src == 3 ) {
      temp = start[0];
      start[0] = '\0';
    } else {
      temp = start[-1];
      start[-1] = '\0';
    }
	if( f_opendir(&dir, src) != FR_OK ) {
      start[start - src == 3 ? 0 : -1] = temp;
      return 0;
    }   
    start[start - src == 3 ? 0 : -1] = temp;

    do {
      if( f_readdir(&dir, &file) != FR_OK )
        return 0;

      len = strlen(file.fname);
      if( (len                 == ptr-start && !strncmp(file.fname, start, ptr-start)) ||
          (strlen(file.lfname) == ptr-start && !strncasecmp(file.lfname, start, ptr-start)) ) {
        if( max_dest_len < len+1 )
          return 0;
        memcpy(dest, file.fname, len);
        dest += len;
        if( *ptr == '\0' ) {
          *dest = '\0';
          return 1;
        }
        *dest++ = '\\';
        max_dest_len -= len+1;
        ++ptr;
        break;
      }
    } while( dir.sect != 0 );
    if( !dir.sect )
      return 0;
  }
  return 1;
}

static void Load_Config_File() {
  unsigned char basstreble_updated = 0;
  char buf[256+2];

  DACConfig InitState = g_Config;
  buf[0] = '0';
  buf[1] = ':';
  buf[2] = '\0';
  find_and_read_config(&g_Config, buf+2, sizeof(buf)-2);
  if( buf[2] != '\0' && lfn_to_sfn(resume_path, buf, sizeof(resume_path)) ) {
    resume_state = g_Config.Auto_Play ? playing : stopped;
    resume_pos = 0;
  } else {
    if( !g_Config.Auto_Play && (resume_state == playing || resume_state == paused) )
      resume_state = stopped;
  }
  if( g_Config.Mute != InitState.Mute ) {
    if( g_Config.Mute ) {
      if( ++muted == 1 )
        UpdateMuteStatus();
    } else {
      if( --muted == 0 )
        UpdateMuteStatus();
    }
  }
  if( g_Config.Volume != InitState.Volume || g_Config.Balance != InitState.Balance || (g_Config.Tone_Enabled && g_Config.Tone_AutoLoudness) != (InitState.Tone_Enabled && InitState.Tone_AutoLoudness) ) {
    if( g_Config.Tone_AutoLoudness ) {
      basstreble_updated = 1;
      if( !UpdateBassTreble() )
        CS4398_Set_Volume_and_Balance(g_Config.Volume + g_CurrentVolumeAdjust, g_Config.Balance);
    } else {
      CS4398_Set_Volume_and_Balance(g_Config.Volume + g_CurrentVolumeAdjust, g_Config.Balance);
    }
  }
  Playback_Set_Order(g_Config.PlaybackOrder);
  if( !basstreble_updated && (g_Config.Tone_Enabled             != InitState.Tone_Enabled ||
                              g_Config.Tone_EqualVolume         != InitState.Tone_EqualVolume ||
                              g_Config.Tone_BassBoostCut        != InitState.Tone_BassBoostCut ||
                              g_Config.Tone_TrebleBoostCut      != InitState.Tone_TrebleBoostCut ||
                              g_Config.Tone_BassCrossoverFreq   != InitState.Tone_BassCrossoverFreq ||
                              g_Config.Tone_TrebleCrossoverFreq != InitState.Tone_TrebleCrossoverFreq) )
    UpdateBassTreble();

  if( g_Config.Crossfeed_Enabled != InitState.Crossfeed_Enabled ||
      g_Config.Crossfeed_LPFFreq != InitState.Crossfeed_LPFFreq ||
      g_Config.Crossfeed_Delay   != InitState.Crossfeed_Delay   ||
      g_Config.Crossfeed_Atten   != InitState.Crossfeed_Atten )
    UpdateCrossfeed();
  
  if( g_Config.DownmixToMono  != InitState.DownmixToMono ||
      g_Config.StereoSwap     != InitState.StereoSwap ||
      g_Config.InvertPolarity != InitState.InvertPolarity )
    CS4398_Set_Channel_Config();

  if( g_Config.Filter_Slowrolloff != InitState.Filter_Slowrolloff )
    CS4398_Set_Filter_Config();

  if( g_Config.NoDeEmphasis != InitState.NoDeEmphasis )
    CS8416_Update_Deemph_Config();

  if( g_Config.NoFreeRunningPLL != InitState.NoFreeRunningPLL )
    CS8416_Update_RMCKF_and_SWCLK();
}

//////////// Functions and variables for dealing with LEDs on front panel ////////////

unsigned char g_ChannelStates[8], g_LastChannelStates[8];

static void SetInputLED(unsigned char which, unsigned char bright) {
  TRISB = (TRISB & ~(3<<(which<<1))) | ((3-bright)<<(which<<1));
}
#define DIM    1
#define BRIGHT 2
#define FLASH  3
#define PULSE  4
static unsigned char Update_Sampling_Rate_LEDs(unsigned long rate, unsigned char deemph, unsigned char fmt_regB, unsigned char err_regC, unsigned char* valid_audio) {
  unsigned char leds, mode;
  TRISDbits.TRISD10 = 0;
  TRISDbits.TRISD9 = 0;
  TRISCbits.TRISC15 = 0;
  TRISCbits.TRISC12 = 0;
  LATDbits.LATD11 = 1;
  LATDbits.LATD8  = 1;
  *valid_audio = 0;
  if( (err_regC&CS8416_RERR_UNLOCK) ) {
    leds = 0;
    mode = DIM;
  } else if( (err_regC&(CS8416_RERR_BIP|CS8416_RERR_UNLOCK|CS8416_RERR_CCRC|CS8416_RERR_QCRC)) ) {
    leds = 0x0F;
    mode = FLASH;
  } else if( (fmt_regB&CS8416_AUDIO_FORMAT_PCM) ) {
    unsigned char audio = !(fmt_regB&CS8416_AUDIO_FORMAT_DGTL_SIL);
    *valid_audio = audio;
    if( g_Config.Input == 7 ) {
      playback_state_t status = Playback_Get_Status();
      if( status != playing && status != paused )
        rate = 0;
    }
    switch(rate) {
    case 0:
      leds = 0;
      mode = DIM;
      break;
    case 44100UL:
      leds = 1<<0;
      mode = audio ? (deemph ? PULSE : BRIGHT) : DIM;
      break;
    default:
      if( g_ChannelStates[g_Config.Input] ) {
        leds = (1<<0)|(1<<1);
        mode = audio ? (deemph ? PULSE : BRIGHT) : DIM;
      } else {
        leds = 0;
        mode = BRIGHT;
      }
      break;
    case 48000UL:
      leds = 1<<1;
      mode = audio ? (deemph ? PULSE : BRIGHT) : DIM;
      break;
    case 88200UL:
      leds = (1<<1)|(1<<2);
      mode = audio ? BRIGHT : DIM;
      break;
    case 96000UL:
      leds = 1<<2;
      mode = audio ? BRIGHT : DIM;
      break;
    case 176400UL:
      leds = (1<<2)|(1<<3);
      mode = audio ? BRIGHT : DIM;
      break;
    case 192000UL:
      leds = (1<<3);
      mode = audio ? BRIGHT : DIM;
      break;
    }
    if( mode == BRIGHT && (err_regC&(CS8416_RERR_PAR|CS8416_RERR_CONF)) )
      mode = FLASH;
  } else if( fmt_regB&(CS8416_AUDIO_FORMAT_DTS_LD|CS8416_AUDIO_FORMAT_DTS_CD) ) {
    leds = 3;
    mode = FLASH;
  } else if( fmt_regB&CS8416_AUDIO_FORMAT_IEC61937 ) {
    leds = 6;
    mode = FLASH;
  } else if( (err_regC&CS8416_RERR_VALID) ) {
    leds = 12;
    mode = FLASH;
  } else {
    leds = 0;
    mode = DIM;
  }
  LATDbits.LATD10 = !(leds&1);
  LATDbits.LATD9  = !((leds>>1)&1);
  LATCbits.LATC15 = !((leds>>2)&1);
  LATCbits.LATC12 = !((leds>>3)&1);
  TRISDbits.TRISD11 = (mode == DIM);;
  TRISDbits.TRISD8 = (mode != DIM && mode != PULSE);
  return mode;
}

//////////// Functions for going into and out of standby ////////////

static void PowerControl(bool bOn) {
  TRISGbits.TRISG3 = 0;
  LATGbits.LATG3 = bOn;
}

#define IR_OPT1 0x00040000L

unsigned short ignore_poweroff_command;
static void PowerOff(bool bSave) {
  unsigned char i;

  if( bSave )
    SaveSettingsToFlash();

  for( i = 0; i < 8; ++i )
    SetInputLED(i, 0);
  memoryCardSystemUp = 0;
  SetSDCardPower(false);
  PLL1708_Configure(0);
  CS4398_HoldInReset();
  CS8416_HoldInReset();
  Clock_SetSpeed(false);
  Timer_Delay_ms(250);
  PowerControl(false);

  // turn off sampling rate lEDs
  LATDbits.LATD10 = 1;
  LATDbits.LATD9  = 1;
  LATCbits.LATC15 = 1;
  LATCbits.LATC12 = 1;
  TRISDbits.TRISD11 = 0;
  TRISDbits.TRISD8 = 0;

  // enable wake up via front panel power swith
  CNPU2bits.CN16PUE = 1;
  while( !PORTDbits.RD7 )
    Nop();
  Timer_Delay_ms(10); // wait for switch contacts to stop bouncing
  IFS1bits.CNIF = 0;
  CNEN2bits.CN16IE = 1;

  DCICON1bits.DCIEN = 0;

  TRISGbits.TRISG0 = 1;

  i = 0;
  while(1) {
    if( i == 0 ) {
      Sleep();
    } else {
      --i;
    }
    Timer_Delay_ms(25);
    if( !PORTDbits.RD7 )
      break; // power button pushed
    if( ir_final_code ) {
      unsigned long ir_code = ir_final_code;
      ir_final_code = 0;
      if( g_Config.AlternativeIRCodes )
        ir_code |= IR_OPT1;
      if( ir_code == (IR_RC5|RC5_A1012_TV156_STANDBY) ||
          ir_code == (IR_RC5|RC5_A1012_TV156_STANDBY|IR_REPEAT) ||
          ir_code == (IR_RC5|RC5_AR1729_252_STANDBY) ||
          ir_code == (IR_RC5|RC5_AR1729_252_STANDBY|IR_REPEAT) ||
          ir_code == (IR_RC5|RC5_A1012_TV170_STANDBY|IR_OPT1) ||
          ir_code == (IR_RC5|RC5_A1012_TV170_STANDBY|IR_OPT1|IR_REPEAT) ||
          ir_code == (IR_NEC|NEC_AR1729_281_STANDBY|IR_OPT1) ||
          ir_code == (IR_NEC|NEC_AR1729_281_STANDBY|IR_OPT1|IR_REPEAT) ) {
        if( i > 0 ) {
          ignore_poweroff_command = 8/*4000*/;
          break;
        }
        i = 100;
      }
    }
  }

  CNEN2bits.CN16IE = 0;
  IFS1bits.CNIF = 0;
  CNPU2bits.CN16PUE = 0;
  g_ignore_power_button = 10;

  TRISGbits.TRISG0 = 0;

  DCICON1bits.DCIEN = 1;

  PowerControl(true);
  Timer_Delay_ms(250);
  Clock_SetSpeed(true);
  CS8416_Setup();
  CS4398_Setup();
}

//////////// Rest of the global state variables and pin change notification interrupt handler ////////////

unsigned char g_InitialisingSDCard, g_ScanInput, g_CurrentSpeedMode;
unsigned char g_LastCardState;
unsigned short g_LastCardStateChange;
static const unsigned char g_InputMap[8] = { 4, 5, 6, 0, 1, 2, 3, 7 };
unsigned char g_DIPSwitchStates;
unsigned char ac_sense, ir_timer_on;
typedef enum { normal, basstreble, crossfeed } remote_modes;

// This can be called when the state of one of several pins changes (depending on which notifications are enabled).
// These can signal a change in the received digital audio signal state, insertion or removal of the SD card or a mains zero-crossing event.
void ChangeNotificationInterrupt() {
  static unsigned char last_RD5 = 0;
  if( CNEN1bits.CN14IE && (PORTDbits.RD5 != last_RD5) ) {
    last_RD5 ^= 1;
    CS8416_Read_Register(CS8416_REG_INTERRUPT_STATUS);
    UpdateMuteStatus();
  }
  if( g_LastCardState == PORTFbits.RF4 ) {
    g_LastCardState ^= 1;
    g_LastCardStateChange = TMR4;
    if( PORTFbits.RF4 ) {
      memoryCardSystemUp = false;
      SetSDCardPower(false);
      CNEN2bits.CN17IE = 0;
    }
  }
  if( ac_sense != PORTDbits.RD7 ) {
    ac_sense = PORTDbits.RD7;
    if( !ac_sense ) {
      CNPU2bits.CN16PUE = 1;
      Nop();
      Nop();
      if( !PORTDbits.RD7 ) {
        power_button_pressed = 1;
      } else {
        CNPU2bits.CN16PUE = 0;
        if( TMR5 < 25000 && disable_power_loss_quick_sense >= 25 && T5CONbits.TON )
          PowerLost(3);
      }
    } else if( power_button_pressed ) {
      power_button_pressed = 0;
      CNPU2bits.CN16PUE = 0;
      disable_power_loss_quick_sense = 0;
    } else {
      if( TMR5 < 25000 && disable_power_loss_quick_sense >= 25 && T5CONbits.TON )
        PowerLost(4);
    }
    if( disable_power_loss_quick_sense < 25 ) // this is to allow it to settle down after power-on and initialisation, or after the button is released
      ++disable_power_loss_quick_sense;
    TMR5 = 0;

    if( g_InitialisingSDCard ) {
      SetInputLED(7, (g_Config.Input == 7 ? 2 : 0) | (g_InitialisingSDCard>>4));
      if( ++g_InitialisingSDCard == 32+1 )
        g_InitialisingSDCard = 1;
    }
  }
}

//////////// Main loop ////////////

int main(void) {
  unsigned char mute_timer_on = 0, long_mute_timer = 0, led_mode = DIM, led_flash_timer = 0, USB_audio_present = 0, pwb = 0, vol_timer_on = 0;
  unsigned char curremotemode = normal, ir_repeat_timer_on = 0;
  unsigned short got_poweroff_command = 0;
  unsigned short mute_timer = 0, vol_timer = 0, ir_timer = 0, ir_repeat_timer = 0, no_audio_timer = 0, underrun = 0;
  unsigned long EstimatedRate, RawEstimatedRate, last_ir_code = 0;

  // enable input LEDs
  LATB = 0;

  // set up interrupt priority
  IPC1bits.DMA0IP = 5;  // DCI DMA has highest interrupt priority since S/PDIF generation can't have any hiccups
  IPC2bits.SPI1IP = 6;  // high priority, to allow SPI transfers to go as quickly as possible, without interrupting DCI (priority=7)

  reset_config_to_default(&g_Config);

  // initialise peripherals
  Clock_SetSpeed(true);
  Biphase_Generate_Tables();
  SetSDCardPower(false);
  InitSDCardDetect();
  Timer_Setup();
  SPI_Setup();

  // initialisa DAC and digital audio receiver ICs
  CS4398_HoldInReset();
  CS8416_HoldInReset();
  Timer_Delay_ms(10);
  PowerControl(true);
  Timer_Delay_ms(50);

  DCI_Setup();
  CS8416_Setup();
  CS4398_Setup();

  // do some power-on self testing of some of the chips
  if( !Test_PLL1708() ) {
    while(1) {
      SetInputLED(7, 3);
      Timer_Delay_ms(250);
      SetInputLED(7, 0);
      Timer_Delay_ms(250);
    }
  }
  if( !CS8416_Verify_ID() ) {
    while(1) {
      SetInputLED(6, 3);
      Timer_Delay_ms(250);
      SetInputLED(6, 0);
      Timer_Delay_ms(250);
    }
  }
  if( !CS4398_Verify_ID() ) {
    while(1) {
      SetInputLED(5, 3);
      Timer_Delay_ms(250);
      SetInputLED(5, 0);
      Timer_Delay_ms(250);
    }
  }

  // more initialisation
  DataEEInit();
  LoadSettingsFromFlash();

  g_DIPSwitchStates = 0;

  // read DIP switch #1
  LATFbits.LATF0 = 1;
  TRISFbits.TRISF0 = 0;
  Nop();
  Nop();
  poweron = !PORTFbits.RF0;
  g_DIPSwitchStates |= !PORTFbits.RF0;
  Nop();
  Nop();
  TRISFbits.TRISF0 = 1;
  LATFbits.LATF0 = 0;

  // read DIP switch #2
  LATFbits.LATF1 = 1;
  TRISFbits.TRISF1 = 0;
  Nop();
  Nop();
  g_DIPSwitchStates |= (!PORTFbits.RF1)<<1;
  Nop();
  Nop();
  TRISFbits.TRISF1 = 1;
  LATFbits.LATF1 = 0;

  // read DIP switch #3
  LATGbits.LATG1 = 1;
  TRISGbits.TRISG1 = 0;
  Nop();
  Nop();
  g_DIPSwitchStates |= (!PORTGbits.RG1)<<2;
  Nop();
  Nop();
  TRISGbits.TRISG1 = 1;
  LATGbits.LATG1 = 0;

  // reset configuration again so that DIP switch readings can take effect
  reset_config_to_default(&g_Config);

  // enable power LED
  LATGbits.LATG0 = 0;

  // if starting up in standby mode (DIP switch #1), go into standby now
  if( !poweron ) {
    Infrared_Setup();
    // make an attempt to load the configuration file
    if( !PORTFbits.RF4 ) {
      memoryCardSystemUp = true;
      g_InitialisingSDCard = 1;
      SetSDCardPower(true);
      CNEN2bits.CN17IE = 1;
      Timer_Delay_ms(10);

      Playback_Reset();
      if( Playback_Setup() == ok )
        Load_Config_File(resume_path);
      g_InitialisingSDCard = 0;
      CNEN2bits.CN17IE = 0;
      SetSDCardPower(false);
      memoryCardSystemUp = false;
    }
    PowerOff(false);
  }

  // turn on power LED
  TRISGbits.TRISG0 = 0;

  // loop runs once for each time the unit goes though a power-up/standby cycle
  while(1) {
    // after power-up we need to configure the digital audio receiver to set the right channel, mode, etc.
    CS8416_Update_Deemph_Config();
    CS8416_Write_Register(CS8416_REG_CONTROL3, (CS8416_GPO_INT<<CS8416_CONTROL3_GPO1SEL_SHIFT) | (CS8416_GPO_GND<<CS8416_CONTROL3_GPO2SEL_SHIFT));
    CS8416_Write_Register(CS8416_REG_CONTROL4, CS8416_CONTROL4_RUN|(g_InputMap[g_Config.Input]<<CS8416_CONTROL4_RXSEL_SHIFT)|(g_InputMap[g_ScanInput]<<CS8416_CONTROL4_TXSEL_SHIFT));
    CS8416_Write_Register(CS8416_REG_SERIAL_DATA_AUDIO_FORMAT, CS8416_SDAF_SOMS);
    CS8416_Write_Register(CS8416_REG_RCVR_ERROR_MASK, CS8416_RERR_UNLOCK|CS8416_RERR_VALID|CS8416_RERR_BIP|CS8416_RERR_PAR);
    CS8416_Write_Register(CS8416_REG_INTERRUPT_MASK, CS8416_INT_RERR|CS8416_INT_FCH);
    CS8416_Write_Register(CS8416_REG_INTERRUPT_MODE_MSB, 0);
    CS8416_Write_Register(CS8416_REG_INTERRUPT_MODE_LSB, CS8416_INT_RERR);
    Timer_Delay_ms(10);

    // same with the DAC
    CS4398_Setup();
    CS4398_Set_Volume_and_Balance(g_Config.Volume + g_CurrentVolumeAdjust, g_Config.Balance);
    CS4398_Set_Channel_Config();
    CS4398_Set_Filter_Config();
    UpdateMuteStatus();

    // now set up IR reception, AC power loss detection, etc.
    ir_cn_interrupt_chain = &ChangeNotificationInterrupt;
    Infrared_Setup();
    ACSense_Init();
    PLL1708_Configure(44100);
    poweron = true;

    DCI_Start_SPDIF_Transmission();

    // main loop, terminates when going into standby
    while(poweron) {
      unsigned char SpeedMode, RMCKF;
      // get the current sampling rate from the digital audio receiver and configure the DAC appropriately (it needs hand-holding)
      EstimatedRate = EstimatePlaybackSampleRate(&RawEstimatedRate);
      if( EstimatedRate > 100000 ) {
        SpeedMode = CS4398_MODE1_FM_QUAD;
        RMCKF = 1/*128*/;
      } else if( EstimatedRate > 50000 ) {
        SpeedMode = CS4398_MODE1_FM_DOUBLE;
        RMCKF = 0/*256*/;
      } else {
        SpeedMode = CS4398_MODE1_FM_SINGLE;
        RMCKF = 0/*256*/;
      }
      if( SpeedMode != g_CurrentSpeedMode ) {
        CS4398_Write_Register(CS4398_REG_MODE1, (SpeedMode<<CS4398_MODE1_FM_SHIFT)|(CS4398_MODE1_DEM_NONE<<CS4398_MODE1_DEM_SHIFT)|(CS4398_MODE1_DIF_LJUST<<CS4398_MODE1_DIF_SHIFT));
        g_CurrentSpeedMode = SpeedMode;
      }
      if( RMCKF != g_CurrentRMCKF ) {
        g_CurrentRMCKF = RMCKF;
        CS8416_Update_RMCKF_and_SWCLK();
      }

      // this timer is used to time LED flashes and so on
      if( IFS1bits.T4IF ) {
        unsigned char valid_audio;

        IFS1bits.T4IF = 0;
        if( ++led_flash_timer == 6 )
          led_flash_timer = 0;

        // updating the sampling rate LEDs periodically allows them to flash or pulse as necessary
        led_mode = Update_Sampling_Rate_LEDs(EstimatedRate, (CS8416_Read_Register(CS8416_REG_RCVR_CHNANEL_STATUS) & CS8416_CHANSTATUS_EMPH) == 0, CS8416_Read_Register(CS8416_REG_AUDIO_FORMAT_DETECT), CS8416_Read_Register(CS8416_REG_RCVR_ERROR), &valid_audio);
        if( led_mode == FLASH ) {
          TRISDbits.TRISD11 = TRISDbits.TRISD8 = (led_flash_timer < 3);
        } else if( led_mode == PULSE ) {
          TRISDbits.TRISD11 =  (led_flash_timer < 3);
          TRISDbits.TRISD8  = !(led_flash_timer < 3);
        }
        if( valid_audio )
          no_audio_timer = 0;
        else
          ++no_audio_timer;
        if( long_mute_timer && !--long_mute_timer ) {
          if( --muted == 0 )
            UpdateMuteStatus();
        }

        // check to see if headphones have been plugged in/removed
        if( g_Config.Crossfeed_Enabled && !g_Config.Crossfeed_IgnoreHPSocket ) {
          unsigned char headphones_plugged_in;
          LATDbits.LATD3 = 1;
          TRISDbits.TRISD3 = 0;
          headphones_plugged_in = PORTDbits.RD3;
          TRISDbits.TRISD1 = 1;
          LATDbits.LATD1 = 0;
          g_Config.Crossfeed_Disabled = !headphones_plugged_in;
        }

        // check to see if it's time to switch to another input (if the current one is inactive and auto-switching is enabled)
        if( no_audio_timer >= g_Config.AutoSwitchDelay*9+(g_Config.AutoSwitchDelay>>1) && g_Config.Auto_Switch_Enabled ) {
          unsigned int i;
          for( i = 1; i < 8; ++i ) {
            if( g_ChannelStates[(g_Config.Input + i)&7] && (g_Config.Input != 7 || Playback_Get_Status() != paused) ) {
              g_Config.Input = (g_Config.Input + i)&7;
              if( !long_mute_timer ) {
                if( ++muted == 1 )
                  UpdateMuteStatus();
              }
              long_mute_timer = 6; // scan rate ~2Hz
              no_audio_timer = g_Config.AutoSwitchDelay*9+(g_Config.AutoSwitchDelay>>1) - (long_mute_timer-1);
              break;
            }
          }
        }

        if( got_poweroff_command )
          --got_poweroff_command;
        if( ignore_poweroff_command )
          --ignore_poweroff_command;
      }

      // indicate DCI buffer underruns by flashing the power LED rapidly
      if( DCIBuffer_underrun && Playback_Get_Status() == playing ) {
        if( (underrun&256) )
          TRISGbits.TRISG0 ^= 1;
        underrun = 4095;
      }
      DCIBuffer_underrun = 0;

      // check to see if a remote control command has been received and if so, decode it and take the necessary action
      if( ir_final_code ) {
        unsigned long ir_code = ir_final_code;
        ir_final_code = 0;
        if( !ir_timer_on )
          TRISGbits.TRISG0 ^= 1;
        ir_timer = TMR4;
        ir_timer_on = 16;
        if( (ir_code|IR_REPEAT) == (last_ir_code|IR_REPEAT) )
          ir_code |= IR_REPEAT;
        last_ir_code = ir_code;

        if( g_Config.AlternativeIRCodes )
          ir_code |= IR_OPT1;

        switch(ir_code) {
        case IR_RC5|RC5_A1012_TV156_STANDBY:
        case IR_RC5|RC5_A1012_TV156_STANDBY|IR_REPEAT:
        case IR_RC5|RC5_AR1729_252_STANDBY:
        case IR_RC5|RC5_AR1729_252_STANDBY|IR_REPEAT:
        case IR_RC5|RC5_A1012_TV170_STANDBY|IR_OPT1:
        case IR_RC5|RC5_A1012_TV170_STANDBY|IR_OPT1|IR_REPEAT:
        case IR_NEC|NEC_AR1729_281_STANDBY|IR_OPT1:
        case IR_NEC|NEC_AR1729_281_STANDBY|IR_OPT1|IR_REPEAT:
          if( got_poweroff_command ) {
            got_poweroff_command = 0;
            poweron = false;
          } else if( !ignore_poweroff_command ) {
            got_poweroff_command = 3/*1500*/;
          } else if( ignore_poweroff_command < 7/*3500*/ ) {
            ignore_poweroff_command += 1/*500*/;
          }
          break;
        case IR_RC5|RC5_A1012_TV156_MUTE:
        case IR_RC5|RC5_AR1729_252_MUTE:
        case IR_RC5|RC5_A1012_TV170_MUTE|IR_OPT1:
        case IR_NEC|NEC_AR1729_281_MUTE|IR_OPT1:
          if( muted == 0 || (muted == 1 && mute_timer_on) ) {
            if( ++muted == 1 )
              UpdateMuteStatus();
            g_Config.Mute = true;
          } else {
            if( --muted == 0 )
              UpdateMuteStatus();
            g_Config.Mute = false;
          }
          break;

        case IR_RC5|RC5_A1012_TV156_1:
        case IR_RC5|RC5_A1012_TV156_2:
        case IR_RC5|RC5_A1012_TV156_3:
        case IR_RC5|RC5_A1012_TV156_4:
        case IR_RC5|RC5_A1012_TV156_5:
        case IR_RC5|RC5_A1012_TV156_6:
        case IR_RC5|RC5_A1012_TV156_7:
        case IR_RC5|RC5_A1012_TV156_8:
        case IR_RC5|RC5_AR1729_252_1:
        case IR_RC5|RC5_AR1729_252_2:
        case IR_RC5|RC5_AR1729_252_3:
        case IR_RC5|RC5_AR1729_252_4:
        case IR_RC5|RC5_AR1729_252_5:
        case IR_RC5|RC5_AR1729_252_6:
        case IR_RC5|RC5_AR1729_252_7:
        case IR_RC5|RC5_AR1729_252_8:
        case IR_RC5|RC5_A1012_TV170_1|IR_OPT1:
        case IR_RC5|RC5_A1012_TV170_2|IR_OPT1:
        case IR_RC5|RC5_A1012_TV170_3|IR_OPT1:
        case IR_RC5|RC5_A1012_TV170_4|IR_OPT1:
        case IR_RC5|RC5_A1012_TV170_5|IR_OPT1:
        case IR_RC5|RC5_A1012_TV170_6|IR_OPT1:
        case IR_RC5|RC5_A1012_TV170_7|IR_OPT1:
        case IR_RC5|RC5_A1012_TV170_8|IR_OPT1:
        case IR_NEC|NEC_AR1729_281_1|IR_OPT1:
        case IR_NEC|NEC_AR1729_281_2|IR_OPT1:
        case IR_NEC|NEC_AR1729_281_3|IR_OPT1:
        case IR_NEC|NEC_AR1729_281_4|IR_OPT1:
        case IR_NEC|NEC_AR1729_281_5|IR_OPT1:
        case IR_NEC|NEC_AR1729_281_6|IR_OPT1:
        case IR_NEC|NEC_AR1729_281_7|IR_OPT1:
        case IR_NEC|NEC_AR1729_281_8|IR_OPT1:
        {
          unsigned long base_key;
          if( ir_code >= (IR_RC5|RC5_A1012_TV156_1) && ir_code <= (IR_RC5|RC5_A1012_TV156_8) )
            base_key = (IR_RC5|RC5_A1012_TV156_1);
          else if( ir_code >= (IR_RC5|RC5_AR1729_252_1) && ir_code <= (IR_RC5|RC5_AR1729_252_8) )
            base_key = (IR_RC5|RC5_AR1729_252_1);
          else if( ir_code >= (IR_RC5|RC5_A1012_TV170_1|IR_OPT1) && ir_code <= (IR_RC5|RC5_A1012_TV170_8|IR_OPT1) )
            base_key = (IR_RC5|RC5_A1012_TV170_1|IR_OPT1);
          else
            base_key = (IR_NEC|NEC_AR1729_281_1|IR_OPT1);
          if( g_Config.Input != ir_code - base_key ) {
            if( !mute_timer_on ) {
              if( ++muted == 1 )
                UpdateMuteStatus();
              mute_timer_on = 1;
            }
            mute_timer = TMR4;
            if( g_Config.Input == 7 )
              DCI_Pause(true);
            g_Config.Input = ir_code - base_key;
            no_audio_timer = 0;
            Timer_Delay_ms(20);
          }
          break;
        }
        case IR_RC5|RC5_A1012_TV156_0:
        case IR_RC5|RC5_AR1729_252_0:
        case IR_RC5|RC5_A1012_TV170_0|IR_OPT1:
        case IR_NEC|NEC_AR1729_281_0|IR_OPT1:
          if( curremotemode == basstreble ) {
            g_Config.Tone_EqualVolume ^= 1;
            UpdateBassTreble();
          }
          break;

        case IR_RC5|RC5_A1012_TV156_CHUP:
        case IR_RC5|RC5_A1012_TV156_CHUP|IR_REPEAT:
        case IR_RC5|RC5_AR1729_252_CHUP:
        case IR_RC5|RC5_AR1729_252_CHUP|IR_REPEAT:
        case IR_RC5|RC5_A1012_TV170_CHUP|IR_OPT1:
        case IR_RC5|RC5_A1012_TV170_CHUP|IR_OPT1|IR_REPEAT:
        case IR_NEC|NEC_AR1729_281_CHUP|IR_OPT1:
        case IR_NEC|NEC_AR1729_281_CHUP|IR_OPT1|IR_REPEAT:
          if( !vol_timer_on ) { // prevents RC5 code corruption from causing a vol up/down button press being interpeted as next/previous folder
            if( curremotemode == normal ) {
              if( g_Config.Balance < +20 ) {
                ++g_Config.Balance;
                CS4398_Set_Volume_and_Balance(g_Config.Volume + g_CurrentVolumeAdjust, g_Config.Balance);
              }
            } else if( curremotemode == basstreble ) {
              if( g_Config.Tone_BassCrossoverFreq < 950 ) {
                g_Config.Tone_BassCrossoverFreq += 50;
                if( g_Config.Tone_BassCrossoverFreq > 950 )
                  g_Config.Tone_BassCrossoverFreq = 950;
                UpdateBassTreble();
              }
            } else if( curremotemode == crossfeed ) {
              if( g_Config.Crossfeed_Atten < 5 ) {
                ++g_Config.Crossfeed_Atten;
                UpdateCrossfeed();
              }
            }
          }
          break;
        case IR_RC5|RC5_A1012_TV156_CHDN:
        case IR_RC5|RC5_A1012_TV156_CHDN|IR_REPEAT:
        case IR_RC5|RC5_AR1729_252_CHDN:
        case IR_RC5|RC5_AR1729_252_CHDN|IR_REPEAT:
        case IR_RC5|RC5_A1012_TV170_CHDN|IR_OPT1:
        case IR_RC5|RC5_A1012_TV170_CHDN|IR_OPT1|IR_REPEAT:
        case IR_NEC|NEC_AR1729_281_CHDN|IR_OPT1:
        case IR_NEC|NEC_AR1729_281_CHDN|IR_OPT1|IR_REPEAT:
          if( !vol_timer_on ) { // prevents RC5 code corruption from causing a vol up/down button press being interpeted as next/previous folder
            if( curremotemode == normal ) {
              if( g_Config.Balance > -20 ) {
                --g_Config.Balance;
                CS4398_Set_Volume_and_Balance(g_Config.Volume + g_CurrentVolumeAdjust, g_Config.Balance);
              }
            } else if( curremotemode == basstreble ) {
              if( g_Config.Tone_BassCrossoverFreq > 50 ) {
                g_Config.Tone_BassCrossoverFreq -= 50;
                if( g_Config.Tone_BassCrossoverFreq < 50 )
                    g_Config.Tone_BassCrossoverFreq = 50;
                UpdateBassTreble();
              }
            } else if( curremotemode == crossfeed ) {
              if( g_Config.Crossfeed_Atten > 1 ) {
                --g_Config.Crossfeed_Atten;
                UpdateCrossfeed();
              }
            } 
          }
          break;

        case IR_RC5|RC5_A1012_TV156_VOLUP:
        case IR_RC5|RC5_A1012_TV156_VOLUP|IR_REPEAT:
        case IR_RC5|RC5_AR1729_252_VOLUP:
        case IR_RC5|RC5_AR1729_252_VOLUP|IR_REPEAT:
        case IR_RC5|RC5_A1012_TV170_VOLUP|IR_OPT1:
        case IR_RC5|RC5_A1012_TV170_VOLUP|IR_OPT1|IR_REPEAT:
        case IR_NEC|NEC_AR1729_281_VOLUP|IR_OPT1:
        case IR_NEC|NEC_AR1729_281_VOLUP|IR_OPT1|IR_REPEAT:
          if( curremotemode == normal || curremotemode == crossfeed ) {
            if( g_Config.Volume > 0 ) {
              --g_Config.Volume;
              if( g_Config.Tone_AutoLoudness ) {
                if( UpdateBassTreble() )
                  break;
              }
              CS4398_Set_Volume_and_Balance(g_Config.Volume + g_CurrentVolumeAdjust, g_Config.Balance);
            }
            vol_timer_on = 0;
            vol_timer = TMR4;
          } else if( curremotemode == basstreble ) {
            if( g_Config.Tone_TrebleCrossoverFreq < 5000 ) {
              g_Config.Tone_TrebleCrossoverFreq += 200;
              if( g_Config.Tone_TrebleCrossoverFreq > 5000 )
                g_Config.Tone_TrebleCrossoverFreq = 5000;
              UpdateBassTreble();
            }
          }
          break;
        case IR_RC5|RC5_A1012_TV156_VOLDN:
        case IR_RC5|RC5_A1012_TV156_VOLDN|IR_REPEAT:
        case IR_RC5|RC5_AR1729_252_VOLDN:
        case IR_RC5|RC5_AR1729_252_VOLDN|IR_REPEAT:
        case IR_RC5|RC5_A1012_TV170_VOLDN|IR_OPT1:
        case IR_RC5|RC5_A1012_TV170_VOLDN|IR_OPT1|IR_REPEAT:
        case IR_NEC|NEC_AR1729_281_VOLDN|IR_OPT1:
        case IR_NEC|NEC_AR1729_281_VOLDN|IR_OPT1|IR_REPEAT:
          if( curremotemode == normal || curremotemode == crossfeed ) {
            if( g_Config.Volume < 255 ) {
              ++g_Config.Volume;
              if( g_Config.Tone_AutoLoudness ) {
                if( UpdateBassTreble() )
                  break;
              }
              CS4398_Set_Volume_and_Balance(g_Config.Volume + g_CurrentVolumeAdjust, g_Config.Balance);
            }
            vol_timer_on = 0;
            vol_timer = TMR4;
          } else if( curremotemode == basstreble ) {
            if( g_Config.Tone_TrebleCrossoverFreq > 1000 ) {
              g_Config.Tone_TrebleCrossoverFreq -= 200;
              if( g_Config.Tone_TrebleCrossoverFreq < 1000 )
                g_Config.Tone_TrebleCrossoverFreq = 1000;
              UpdateBassTreble();
            }
          }
          break;

        case IR_RC5|RC5_A1012_TV156_UP:
        case IR_RC5|RC5_A1012_TV156_UP|IR_REPEAT:
        case IR_RC5|RC5_AR1729_252_UP:
        case IR_RC5|RC5_AR1729_252_UP|IR_REPEAT:
        case IR_RC5|RC5_A1012_TV170_UP|IR_OPT1:
        case IR_RC5|RC5_A1012_TV170_UP|IR_OPT1|IR_REPEAT:
        case IR_NEC|NEC_AR1729_281_UP|IR_OPT1:
        case IR_NEC|NEC_AR1729_281_UP|IR_OPT1|IR_REPEAT:
          if( curremotemode == normal && !(ir_code&IR_REPEAT) ) {
            if( Playback_Get_Status() == playing || Playback_Get_Status() == paused ) {
              if( !mute_timer_on ) {
                if( ++muted == 1 )
                  UpdateMuteStatus();
                mute_timer_on = 1;
              }
              mute_timer = TMR4;
            }
            Playback_Stop();
            Playback_Prev_Folder();
            Playback_Start();
          } else if( curremotemode == basstreble ) {
            if( g_Config.Tone_AutoLoudness || g_Config.Tone_BassBoostCut < 16 ) {
              ++g_Config.Tone_BassBoostCut;
              g_Config.Tone_AutoLoudness = 0;
              UpdateBassTreble();
            }
          } else if( curremotemode == crossfeed ) {
            if( g_Config.Crossfeed_Delay < 31 ) {
              ++g_Config.Crossfeed_Delay;
              UpdateCrossfeed();
            }
          }
          break;
        case IR_RC5|RC5_A1012_TV156_DOWN:
        case IR_RC5|RC5_A1012_TV156_DOWN|IR_REPEAT:
        case IR_RC5|RC5_AR1729_252_DOWN:
        case IR_RC5|RC5_AR1729_252_DOWN|IR_REPEAT:
        case IR_RC5|RC5_A1012_TV170_DOWN|IR_OPT1:
        case IR_RC5|RC5_A1012_TV170_DOWN|IR_OPT1|IR_REPEAT:
        case IR_NEC|NEC_AR1729_281_DOWN|IR_OPT1:
        case IR_NEC|NEC_AR1729_281_DOWN|IR_OPT1|IR_REPEAT:
          if( curremotemode == normal && !(ir_code&IR_REPEAT) ) {
            if( Playback_Get_Status() == playing || Playback_Get_Status() == paused ) {
              if( !mute_timer_on ) {
                if( ++muted == 1 )
                  UpdateMuteStatus();
                mute_timer_on = 1;
              }
              mute_timer = TMR4;
            }
            Playback_Stop();
            Playback_Next_Folder();
            Playback_Start();
          } else if( curremotemode == basstreble ) {
            if( g_Config.Tone_AutoLoudness || g_Config.Tone_BassBoostCut > -8 ) {
              --g_Config.Tone_BassBoostCut;
              g_Config.Tone_AutoLoudness = 0;
              UpdateBassTreble();
              CS4398_Set_Volume_and_Balance(g_Config.Volume + g_CurrentVolumeAdjust, g_Config.Balance);
            }
          } else if( curremotemode == crossfeed ) {
            if( g_Config.Crossfeed_Delay > 0 ) {
              --g_Config.Crossfeed_Delay;
              UpdateCrossfeed();
            }
          }
          break;
        case IR_RC5|RC5_A1012_TV156_LEFT:
        case IR_RC5|RC5_A1012_TV156_LEFT|IR_REPEAT:
        case IR_RC5|RC5_AR1729_252_LEFT:
        case IR_RC5|RC5_AR1729_252_LEFT|IR_REPEAT:
        case IR_RC5|RC5_A1012_TV170_LEFT|IR_OPT1:
        case IR_RC5|RC5_A1012_TV170_LEFT|IR_OPT1|IR_REPEAT:
        case IR_NEC|NEC_AR1729_281_LEFT|IR_OPT1:
        case IR_NEC|NEC_AR1729_281_LEFT|IR_OPT1|IR_REPEAT:
          if( curremotemode == normal && !(ir_code&IR_REPEAT) ) {
            if( Playback_Get_Status() == playing || Playback_Get_Status() == paused ) {
              if( !mute_timer_on ) {
                if( ++muted == 1 )
                  UpdateMuteStatus();
                mute_timer_on = 1;
              }
              mute_timer = TMR4;
            }
            Playback_Stop();
            Playback_Prev_Track(true);
            Playback_Start();
          } else if( curremotemode == basstreble ) {
            if( g_Config.Tone_AutoLoudness || g_Config.Tone_TrebleBoostCut > -16 ) {
              --g_Config.Tone_TrebleBoostCut;
              g_Config.Tone_AutoLoudness = 0;
              UpdateBassTreble();
            }
          } else if( curremotemode == crossfeed ) {
            if( g_Config.Crossfeed_LPFFreq > 500 ) {
              g_Config.Crossfeed_LPFFreq -= 100;
              if( g_Config.Crossfeed_LPFFreq < 500 )
                g_Config.Crossfeed_LPFFreq = 500;
              UpdateCrossfeed();
            }
          }
          break;
        case IR_RC5|RC5_A1012_TV156_RIGHT:
        case IR_RC5|RC5_A1012_TV156_RIGHT|IR_REPEAT:
        case IR_RC5|RC5_AR1729_252_RIGHT:
        case IR_RC5|RC5_AR1729_252_RIGHT|IR_REPEAT:
        case IR_RC5|RC5_A1012_TV170_RIGHT|IR_OPT1:
        case IR_RC5|RC5_A1012_TV170_RIGHT|IR_OPT1|IR_REPEAT:
        case IR_NEC|NEC_AR1729_281_RIGHT|IR_OPT1:
        case IR_NEC|NEC_AR1729_281_RIGHT|IR_OPT1|IR_REPEAT:
          if( curremotemode == normal && !(ir_code&IR_REPEAT) ) {
            if( Playback_Get_Status() == playing || Playback_Get_Status() == paused ) {
              if( !mute_timer_on ) {
                if( ++muted == 1 )
                  UpdateMuteStatus();
                mute_timer_on = 1;
              }
              mute_timer = TMR4;
            }
            Playback_Stop();
            Playback_Next_Track(true);
            Playback_Start();
          } else if( curremotemode == basstreble ) {
            if( g_Config.Tone_AutoLoudness || g_Config.Tone_TrebleBoostCut < 16 ) {
              ++g_Config.Tone_TrebleBoostCut;
              g_Config.Tone_AutoLoudness = 0;
              UpdateBassTreble();
            }
          } else if( curremotemode == crossfeed ) {
            if( g_Config.Crossfeed_LPFFreq < 5000 ) {
              g_Config.Crossfeed_LPFFreq += 100;
              if( g_Config.Crossfeed_LPFFreq > 5000 )
                g_Config.Crossfeed_LPFFreq = 5000;
              UpdateCrossfeed();
            }
          }
          break;
        case IR_RC5|RC5_A1012_TV156_OK:
        case IR_RC5|RC5_AR1729_252_OK:
        case IR_RC5|RC5_A1012_TV170_OK|IR_OPT1:
        case IR_NEC|NEC_AR1729_281_OK|IR_OPT1:
          if( curremotemode == normal ) {
            Playback_Set_Order(Playback_Get_Order() == sorted ? shuffle : (Playback_Get_Order() == shuffle ? directory : sorted));
          } else if( curremotemode == basstreble ) {
            g_Config.Tone_BassBoostCut = 0;
            g_Config.Tone_TrebleBoostCut = 0;
            g_Config.Tone_BassCrossoverFreq = 500;
            g_Config.Tone_TrebleCrossoverFreq = 2000;
            g_Config.Tone_AutoLoudness = 0;
            UpdateBassTreble();
          } else if( curremotemode == crossfeed ) {
            g_Config.Crossfeed_LPFFreq = 1500;
            g_Config.Crossfeed_Delay = 14;
            g_Config.Crossfeed_Atten = 2;
            UpdateCrossfeed();
          }
          break;

        case IR_RC5|RC5_A1012_TV156_TELETEXT:
        case IR_RC5|RC5_AR1729_252_TV_VCR:
        case IR_RC5|RC5_A1012_TV170_TELETEXT|IR_OPT1:
        case IR_NEC|NEC_AR1729_281_TV_VCR|IR_OPT1:
          curremotemode = basstreble;
          break;
        case IR_RC5|RC5_A1012_TV156_PAGEHOLD:
        case IR_RC5|RC5_AR1729_252_TV_AV:
        case IR_RC5|RC5_A1012_TV170_PAGEHOLD|IR_OPT1:
        case IR_NEC|NEC_AR1729_281_TV_AV|IR_OPT1:
          curremotemode = crossfeed;
          break;
        case IR_RC5|RC5_A1012_TV156_TVVIDEO:
        case IR_RC5|RC5_AR1729_252_MENU:
        case IR_RC5|RC5_A1012_TV170_TVVIDEO|IR_OPT1:
        case IR_NEC|NEC_AR1729_281_MENU|IR_OPT1:
          curremotemode = normal;
          break;

        case IR_RC5|RC5_A1012_TV156_9:
        case IR_RC5|RC5_AR1729_252_9:
        case IR_RC5|RC5_A1012_TV170_9|IR_OPT1:
        case IR_NEC|NEC_AR1729_281_9|IR_OPT1:
          if( curremotemode == basstreble ) {
              g_Config.Tone_AutoLoudness = 1;
              UpdateBassTreble();
          }
          break;
        case IR_RC5|RC5_A1012_TV156_RECORD:
        case IR_RC5|RC5_AR1729_252_RECORD:
        case IR_RC5|RC5_A1012_TV170_RECORD|IR_OPT1:
        case IR_NEC|NEC_AR1729_281_RECORD|IR_OPT1:
          if( curremotemode == normal ) {
            if( g_Config.Balance != 0 ) {
              g_Config.Balance = 0;
              CS4398_Set_Volume_and_Balance(g_Config.Volume + g_CurrentVolumeAdjust, g_Config.Balance);
            }
          } else if( curremotemode == basstreble ) {
            g_Config.Tone_Enabled ^= 1;
            UpdateBassTreble();
          } else if( curremotemode == crossfeed ) {
            g_Config.Crossfeed_Enabled ^= 1;
            UpdateCrossfeed();
          }
          break;
        case IR_RC5|RC5_A1012_TV156_PAUSE:
        case IR_RC5|RC5_AR1729_252_PAUSE:
        case IR_RC5|RC5_A1012_TV170_PAUSE|IR_OPT1:
        case IR_NEC|NEC_AR1729_281_PAUSE|IR_OPT1:
          if( Playback_Get_Status() == paused )
            Playback_Start();
          else
            Playback_Pause();
          break;

        case IR_RC5|RC5_A1012_TV156_RW:
        case IR_RC5|RC5_A1012_TV156_RW|IR_REPEAT:
        case IR_RC5|RC5_AR1729_252_RW:
        case IR_RC5|RC5_AR1729_252_RW|IR_REPEAT:
        case IR_RC5|RC5_A1012_TV170_RW|IR_OPT1:
        case IR_RC5|RC5_A1012_TV170_RW|IR_OPT1|IR_REPEAT:
        case IR_NEC|NEC_AR1729_281_RW|IR_OPT1:
        case IR_NEC|NEC_AR1729_281_RW|IR_OPT1|IR_REPEAT:
          if( Playback_Get_Status() == playing || Playback_Get_Status() == paused )
            Playback_Back();
          break;
        case IR_RC5|RC5_A1012_TV156_PLAY:
        case IR_RC5|RC5_AR1729_252_PLAY:
        case IR_RC5|RC5_A1012_TV170_PLAY|IR_OPT1:
        case IR_NEC|NEC_AR1729_281_PLAY|IR_OPT1:
          if( Playback_Get_Status() == paused ) {
            Playback_Start();
          } else if( Playback_Get_Status() != playing ) {
            DCI_Pause(true);
            DCI_Reset_Buffer();
            if( Playback_Get_Status() != stopped ) {
              Playback_Reset();
              Playback_Go_To_First_file();
            }
            if( /*Playback_Setup() == ok && */Playback_Start() == ok ) {
              g_Config.Input = 7;
              no_audio_timer = 0;
            }
          }
          break;
        case IR_RC5|RC5_A1012_TV156_FF:
        case IR_RC5|RC5_A1012_TV156_FF|IR_REPEAT:
        case IR_RC5|RC5_AR1729_252_FF:
        case IR_RC5|RC5_AR1729_252_FF|IR_REPEAT:
        case IR_RC5|RC5_A1012_TV170_FF|IR_OPT1:
        case IR_RC5|RC5_A1012_TV170_FF|IR_OPT1|IR_REPEAT:
        case IR_NEC|NEC_AR1729_281_FF|IR_OPT1:
        case IR_NEC|NEC_AR1729_281_FF|IR_OPT1|IR_REPEAT:
          if( Playback_Get_Status() == playing || Playback_Get_Status() == paused )
            Playback_Forward();
          break;
        case IR_RC5|RC5_A1012_TV156_STOP:
        case IR_RC5|RC5_AR1729_252_STOP:
        case IR_RC5|RC5_A1012_TV170_STOP|IR_OPT1:
        case IR_NEC|NEC_AR1729_281_STOP|IR_OPT1:
          DCI_Pause(true);
          if( Playback_Get_Status() == playing || Playback_Get_Status() == paused )
            Playback_Stop();
          else
            Playback_Reset();
          break;
        }
      } else if( mute_timer_on ) {
        // DAC output is sometimes muted briefly when switching inputs and so on - see if it's time to unmute
        unsigned long TimeSinceMute = 65536UL + TMR4 - mute_timer;
        if( TimeSinceMute > 65535UL )
          TimeSinceMute -= 65536UL;
        if( TimeSinceMute > 10000 ) {
          mute_timer_on = 0;
          if( --muted == 0 )
            UpdateMuteStatus();
        }
      } else if( power_button_pressed ) {
        if( g_ignore_power_button ) {
          --g_ignore_power_button;
        } else {
          // If power button is held down, go into standby. If pressed briefly, switch to the next input.
          if( power_button_press_cycles > 100 ) {
            poweron = false;
            pwb = 0;
          } else if( power_button_press_cycles > 5 ) {
            pwb = 1;
          }
        }
      } else if( pwb ) {
        power_button_press_cycles = 0;
        pwb = 0;

        if( !mute_timer_on ) {
          if( ++muted == 1 )
            UpdateMuteStatus();
          mute_timer_on = 1;
        }
        mute_timer = TMR4;
        if( g_Config.Input == 7 )
          DCI_Pause(true);
        g_Config.Input = (g_Config.Input+1)&7;
        no_audio_timer = 0;
        Timer_Delay_ms(20);
      }
      // See if various timers have expired and if so, take appropritae action
      if( ir_timer_on ) {
        unsigned long TimeSinceIRFlash = 65536UL + TMR4 - ir_timer;
        if( TimeSinceIRFlash > 65535UL )
          TimeSinceIRFlash -= 65536UL;
        if( TimeSinceIRFlash > ir_timer_on*1024 ) {
          TRISGbits.TRISG0 ^= 1;
          ir_timer_on = 0;
          ir_repeat_timer = TMR4;
          ir_repeat_timer_on = 16;
        }
      }
      if( ir_repeat_timer_on ) {
        unsigned long TimeSinceIRFlash = 65536UL + TMR4 - ir_repeat_timer;
        if( TimeSinceIRFlash > 65535UL )
          TimeSinceIRFlash -= 65536UL;
        if( TimeSinceIRFlash > ir_repeat_timer_on*1024 ) {
          ir_repeat_timer_on = 0;
          last_ir_code = 0;
        }
      }
      if( vol_timer_on ) {
        unsigned long TimeSinceVol = 65536UL + TMR4 - vol_timer;
        if( TimeSinceVol > 65535UL )
          TimeSinceVol -= 65536UL;
        if( TimeSinceVol > 10000 )
          vol_timer_on = 0;
      }
      if( underrun ) {
        if( (--underrun&255) == 0 )
          TRISGbits.TRISG0 ^= 1;
      }

      // Check to see if SD card has been inserted/removed. If so, initiaise/deinitialise WAV file playback.
      CNEN1bits.CN14IE = 1;
      if( memoryCardSystemUp ) {
        if( g_Config.Input != 7 ) {
          DCI_Pause(true);
        } else {
          if( Playback_Get_Status() == playing && !DCI_Is_Running() )
            DCI_Pause(false);
          Playback_Process();
        }
      }
      // Check to see if the device has been plugged into a computer via USB and if so, switch to that input.
      if( PORTDbits.RD6 && !USB_audio_present ) {
        USB_audio_present = 1;
        if( g_Config.Input != 3 ) {
          if( !mute_timer_on ) {
            if( ++muted == 1 )
              UpdateMuteStatus();
            mute_timer_on = 1;
          }
          mute_timer = TMR4;
          if( g_Config.Input == 7 )
            DCI_Pause(true);
          g_Config.Input = 3;
          no_audio_timer = 0;
          Timer_Delay_ms(20);
        }
      } else if( !PORTDbits.RD6 ) {
        USB_audio_present = 0;
      }
      // More SD card state detection
      if( g_LastCardState == PORTFbits.RF4 ) {
        g_LastCardState ^= 1;
        g_LastCardStateChange = TMR4;
        if( PORTFbits.RF4 ) {
          memoryCardSystemUp = false;
          SetSDCardPower(false);
          CNEN2bits.CN17IE = 0;
        }
      } else if( g_LastCardState && !memoryCardSystemUp ) {
        signed long TimeSinceChange = (signed long)TMR4 - (signed long)g_LastCardStateChange;
        if( TimeSinceChange < 0 )
          TimeSinceChange += 65536UL;
        // A short time after SD card is inserted, configure it, read configuration file and possibly start playback.
        if( TimeSinceChange > 24000 ) {
          unsigned char old_input = g_Config.Input;

          memoryCardSystemUp = true;
          SetSDCardPower(true);
          SetInputLED(7, (g_Config.Input == 7 ? 2 : 0) | 1);
          g_InitialisingSDCard = 1;
          CNEN2bits.CN17IE = 1;
          Timer_Delay_ms(10);

          Playback_Reset();
          DCI_Reset_Buffer();
          g_Config.Input = 7;
          no_audio_timer = 0;
          if( Playback_Setup() == ok ) {
            Playback_Set_Order(g_Config.PlaybackOrder);
            Load_Config_File(resume_path);
			if( g_Config.PlaybackOrder != directory ) {
				Playback_Reset();
				Playback_Go_To_First_file();
			}
            if( (resume_state == playing || resume_state == paused || resume_state == stopped) ) {
              if( Playback_Set_Current_File(resume_path, false) == ok ) {
                Playback_Set_Current_Position(resume_pos);
              } else {
                Playback_Reset();
                Playback_Go_To_First_file();
              }
              if( resume_state == stopped || Playback_Start() == ok ) {
                if( resume_state == paused )
                  Playback_Pause();
                if( !mute_timer_on ) {
                  if( ++muted == 1 )
                    UpdateMuteStatus();
                  mute_timer_on = 1;
                }
                mute_timer = TMR4;
              }
              resume_state = reset;
            } else if( !g_Config.Auto_Play || /*Playback_Setup() != ok || */Playback_Start() != ok ) {
              if( g_Config.Input == 7 )
                g_Config.Input = old_input;
            }
          }
          g_InitialisingSDCard = 0;
        }
      }
      if( !memoryCardSystemUp && Playback_Get_Status() != reset )
        Playback_Reset();

      TMR2 = 0;
      Timer_Delay_us(100);
      CNEN1bits.CN14IE = 0;

      g_ChannelStates[g_ScanInput] = (TMR2 > 50);
      SetInputLED(g_ScanInput, (g_ScanInput == g_Config.Input ? 2 : 0) | (g_ChannelStates[g_ScanInput] ? 1 : 0));
      g_ScanInput = (g_ScanInput+1) & 7;
      if( g_ScanInput == 3 ) {
        g_ChannelStates[3] = PORTDbits.RD6;
        SetInputLED(g_ScanInput, (g_ScanInput == g_Config.Input ? 2 : 0) | (g_ChannelStates[g_ScanInput] ? 1 : 0));
        g_ScanInput = 4;
      } else if( g_ScanInput == 7 ) {
        g_ChannelStates[7] = memoryCardSystemUp;
        SetInputLED(g_ScanInput, (g_ScanInput == g_Config.Input ? 2 : 0) | (g_ChannelStates[g_ScanInput] ? 1 : 0));
        g_ScanInput = 0;
 
        if( !g_ChannelStates[g_Config.Input] && g_LastChannelStates[g_Config.Input] && g_Config.Auto_Switch_Enabled ) {
          unsigned int i;
          for( i = 1; i < 8; ++i ) {
            if( g_ChannelStates[(g_Config.Input + i)&7]/* && !g_LastChannelStates[i]*/ ) {
              g_Config.Input = (g_Config.Input + i)&7;
              no_audio_timer = 0;
              break;
            }
          }
        }
        memcpy(g_LastChannelStates, g_ChannelStates, sizeof(g_LastChannelStates));
      }

      CS8416_Write_Register(CS8416_REG_CONTROL4, CS8416_CONTROL4_RUN|(g_InputMap[g_Config.Input]<<CS8416_CONTROL4_RXSEL_SHIFT)|(g_InputMap[g_ScanInput]<<CS8416_CONTROL4_TXSEL_SHIFT));
    }
    ACSense_Deinit();
    PowerOff(true);
    ir_timer_on = 0;
    last_ir_code = 0;
  }
  return 0;
}

#ifdef __DEBUG
void playback_error() {
    asm volatile("nop");
}
#endif
